/*
	File: CEG_monitor.sqf
	Version: 1.1
	Author: Loyalguard
	
	Description:
	A script to monitor damage and power "in" and "out" status of pertinent CEG  objects.  When a change is detected it will pass the relevant information to a PVEH running on the clients or directly to CEG_event.sqf.

	Parameters:
	_this select 0: The class name of the CEG object (string).
	_this select 1: The logic assigned to the CEG object (object).
	_this select 2: The CEG object closest to the logic (string).
	_this select 3: Other objects/nodes that this object supplies power to (array).
	
	Execution: 
	Executed from CEG_server.sqf.  Will only run in SP, on a MP host, or on a dedicated server due to conditions checked in CEG_server.sqf.  Multiple instances of this script will be running concurrently, one for every logic used by the simulation.
*/

if (!isNil "CEG_DEBUG") then {_debug = ["CEG_monitor.sqf: Thread started for:", (_this select 2)] call LGD_fnc_debugMessage;}; //DEBUG 
 
// Scope //
private ["_class", "_logic", "_obj", "_supplies", "_onStr", "_sound1Str", "_sound1", "_sound2Str", "_sound2", "_pos", "_group"];

// Parameter(s) //
_class = (_this select 0);
_logic = (_this select 1);
_obj = (_this select 2);
_supplies = (_this select 3);

// Run a Continuous loop to check operational status, damage, power in, and power out status of each pertinent CEG object.  If there is a relevant change in status of any of the conditions then broadcast them to the clients (for action by a PVEH) or directly execute and pass them to CEG_events.sqf if the machine is SP or a MP Host.
while {true} do
{
	// Assign local variables for this iteration of the while loop (pertaining to sound logics).
	_sound1Str = (format ["CEG_sound_1_%1", _obj]);
	_sound1 = missionNameSpace getVariable _sound1Str;  // The  first "sound" logic assignd to this CEG object.
	_sound2Str = (format ["CEG_sound_2_%1", _obj]);
	_sound2 = missionNameSpace getVariable _sound2Str;  // The second "sound" logic assignd to this CEG object.
	_pos = getPos _sound1; // The shared position of the sound logics (needed to re-create if deleted when power goes out).
	_group = group _sound1; // The shared group assigned to the sound logics (needed to re-create if deleted when power goes out).

	//// POWER OFF CONDITIONS ////
	
	// Pause the loop until the object is NOT "operational", 100% DAMAGED, it is NOT receiving power (in), OR it is NOT sending power (out).   
	waitUntil
	{
		
		(!(_logic getVariable "operational"))
		or
		(getDammage (position _logic nearestObject _class)  == 1) 
		or
		((missionNameSpace getVariable format ["CEG_in_%1", _obj]) == 0)
		or
		(!(missionNameSpace getVariable format ["CEG_out_%1", _obj]))
	};

	// If the object is destroyed ensure that it is considered NOT operational.
	
	if ((getDammage (position _logic nearestObject _class)  == 1)) then
	{
		_logic setVariable ["operational", false, true];
	};
	
	if (!isNil "CEG_DEBUG") then {_debug = ["CEG_monitor.sqf: Power OFF Condidtions for:", (_this select 2)] call LGD_fnc_debugMessage;}; //DEBUG 
	
	// Change the object's power status to OFF (false).
	_onStr = format ["CEG_on_%1", _obj];
	missionNameSpace setVariable [_onStr, false];

	// Delete the sound logics to immediately stop the sound loop.
	deleteVehicle _sound1;
	deleteVehicle _sound2;
	
	// If the machine is a server then public variable the object's power status to all clients so the relevant PVEH will fire.
	if (isServer) then
	{
		publicVariable _onStr;
	};
	 
	// If the machine is SP or a MP host then manually send the public variable that stores the power status of an object to CEG_events.sqf to ensure that its processes the latest power-related conditions.  This is required since PVEHs don't fire on machines were the PV is broadcast from.
	if ((isServer) and (!isDedicated)) then
	 {
		 _nul = [_onStr, false] execVM "CEG\Scripts\CEG_events.sqf";
	 };
	
	// If this object supplies power to other nodes then spawn an instance of a function to adjust incoming power at those nodes.
	if ((count _supplies) > 0) then {[false, _supplies] spawn CEG_fnc_supply;};
	
	// Momentarily pause the loop to ensure all conditions are updated.
	sleep .01;
	
	
	//// POWER ON CONDITIONS ////
	
	// Pause the loop until the object is "operational", less than 100% DAMAGE, it is RECEIVING power (in), AND it is SENDING power (out).   
	waitUntil
	{
		(_logic getVariable "operational")
		and
		(getDammage (position _logic nearestObject _class) < 1) 
		and
		((missionNameSpace getVariable format ["CEG_in_%1", _obj]) > 0)
		and
		((missionNameSpace getVariable format ["CEG_out_%1", _obj]))
	};
    
	if (!isNil "CEG_DEBUG") then {_debug = ["CEG_monitor.sqf: Power ON Condidtions for:", (_this select 2)] call LGD_fnc_debugMessage;}; //DEBUG 
	
	// Recreate the sound logics so that the sound loop can begin again.
	"Logic" createUnit [_pos, _group, (format ["CEG_sound_1_%1 = this; this setPos [%2, %3, %4];", _obj, (_pos select 0), (_pos select 1), (_pos select 2)])];
	
	"Logic" createUnit [_pos, _group, (format ["CEG_sound_2_%1 = this; this setPos [%2, %3, %4];", _obj, (_pos select 0), (_pos select 1), (_pos select 2)])];
	
	// Change the object's power status to ON (true).
	_onStr = format ["CEG_on_%1", _obj];
	missionNameSpace setVariable [_onStr, true];

	// If the machine is a server then public variable the object's power status to all clients so the relevant PVEH will fire.
	if (isServer) then
	{
		publicVariable _onStr;
	};
	 
	// If the machine is SP or a MP host then manually send the public variable that stores the power status of an object to CEG_events.sqf to ensure that its processes the latest power-related conditions.  This is required since PVEHs don't fire on machines were the PV is broadcast from.
	if ((isServer) and (!isDedicated)) then
	 {
		 _nul = [_onStr, true] execVM "CEG\Scripts\CEG_events.sqf";
	 };
	 
	// If this object supplies power to other nodes then spawn an instance of a function to adjust incoming power at those nodes.
	if ((count _supplies) > 0) then {[true, _supplies] spawn CEG_fnc_supply;};
	
	// Momentarily pause the loop before repeating.
    sleep .01;
	
	// Start next iteration of the loop.
	
}; // End While-Loop Block (But Loop Never Ends!)